# coding=utf-8
from GamePlay import PYBaseGamePlay
from random import choice
from copy import copy
import sys # debugging stuffs

import GEEntity, GEPlayer, GEUtil, GEWeapon, GEMPGameRules, GEGlobal

class Assassination(PYBaseGamePlay):
	DIST_UPDATE_INTERVAL = 0.25 # Time in seconds to which the huds are updated
	COLOUR_TARGET_MSG = GEUtil.CColor(244,192,11,255) # Good kills
	COLOUR_ASSASSIN_MSG = GEUtil.CColor(94,171,231,255) # Better kills
	COLOUR_BADKILL_MSG = GEUtil.CColor(206,43,43,255) # Bad kills
	# Character name dictionary.  This is for the names in the target hud text
	CHARACTER_DICT = {
		'006_mi6':'Trevelyan', 'baron_samedi':'Samedi',
		'bond':'Bond', 'boris':'Boris', 
		'female_scientist':'Scientist', 'jaws':'Jaws', 
		'mayday':'Mayday', 'mishkin':'Mishkin', 
		'oddjob':'Oddjob', 'ourumov':'Ourumov', 
		'valentin':'Valentin'
	}
	# For when player progress bars support localizations
#	CHARACTER_DICT = {
#		'006_mi6':'#GE_006_MI6_CONCISE', 'baron_samedi':'#GE_SAMEDI_CONCISE',
#		'bond':'#GE_BOND_CONCISE', 'boris':'#GE_BORIS_CONCISE', 
#		'female_scientist':'#GE_FEM_SCIENTIST_CONCISE', 'jaws':'#GE_JAWS_CONCISE', 
#		'mayday':'#GE_MAYDAY_CONCISE', 'mishkin':'#GE_MISHKIN_CONCISE', 
#		'oddjob':'#GE_ODDJOB_CONCISE', 'ourumov':'#GE_OURUMOV_CONCISE', 
#		'valentin':'#GE_VALENTIN_CONCISE'
#	}
	
	def __init__(self):
		super(Assassination, self).__init__()
		self.WaitingForPlayers = True
		self.AvaliableTargets = []
		self.AssassinationTargets = [-1 for i in range(32)]
		self.BadKills = [0 for i in range(32)]
		self.BadKillsTime = [0 for i in range(32)]
		self.NextDistUpdate = 0.0
		self.LastDistance = [0 for i in range(32)]
		
	def GetIdent(self):
		return "Assassination"
		
	def GetPrintName(self):
		return "Assassination"
		
	def GetHelpString(self):
		return "Kill your target to earn points while avoiding killing others to \
lose points.  Kill the assassin who has you as a target to gain twice the \
points.  Used a silenced or melee weapon to earn more points!\
\n\nTeam Play: Never\n\nCoded by: Yeyinde"
		
	def GetGameDescription(self):
		return "Assassination"
		
	def GetTeamPlay(self):
		return GEGlobal.TEAMPLAY_NONE
		
	def OnLoadGamePlay(self):
		self.CreateCVar("ass_scorelimit", "0", "Sets a score limit to the assassination game")
		self.CreateCVar("ass_silencedbonus", "2", "Sets a score multiplier when kills are made with silenced weapons")
		self.CreateCVar("ass_meleebonus", "3", "Sets a score multiplier when kills are made with melee weapons")
		self.CreateCVar("ass_badkills", "5", "Sets the bad kill threshold.  Passing this will punish players.")
		self.CreateCVar("ass_punishtime", "15", "Sets the time of no damage punishment.")
		self.CreateCVar("ass_debug", "0", "Assassination debugging verbosity.  Prepare for chatspam!")
		# Always show the radar, because this game mode needs it (When the right funcitons are implemented)
		GEMPGameRules.GetRadar().SetForceRadar(True)
		self.LoadConfig()
		# There is enough players to warrant no wait time, from doing a game mode switch
		if GEMPGameRules.GetNumActivePlayers() >= 2:
			self.WaitingForPlayers = False
		
	def OnRoundBegin(self):
		# Reset scores and regenerate targets on new rounds
		GEMPGameRules.ResetAllPlayersScores()
		self.MassGenerateTargets()
		
	def OnRoundEnd(self):
		self.AvaliableTargets = []
		self.AssassinationTargets = [-1 for i in range(32)]
		self.BadKills = [0 for i in range(32)]
		self.BadKillsTime = [0 for i in range(32)]
		self.LastDistance = [0 for i in range(32)]
		
	def ClientConnect(self, player):
		self.DEBUG("%s connected" % (player.GetPlayerName()))
		if not player:
			return
		self.AssassinationTargets[player.GetIndex()] = -1
		self.BadKills[player.GetIndex()] = 0
		self.BadKillsTime[player.GetIndex()] = 0
		
	def OnPlayerTeamChange(self, player, oldteam, newteam):
		if not player:
			return
		# Add or remove player from avaliable targets
		self.DEBUG("%s changed teams: %s to %s" % (player.GetPlayerName(), oldteam, newteam))
		if newteam == GEGlobal.TEAM_SPECTATOR:
			self.RemoveTarget(player)
		elif oldteam == GEGlobal.TEAM_SPECTATOR:
			self.AddTarget(player)
		
	def ClientDisconnect(self, player):
		self.DEBUG("%s disconnected" % (player.GetPlayerName()))
		if not player:
			return
		# Remove player from avaliable targets
		self.RemoveTarget(player)
		
	def OnPlayerKilled(self, victim, killer, weapon):
		self.DEBUG("%s died" % (victim.GetPlayerName()))
		if not victim:
			return
		point = 1
		# If a weapon kill was made, multiply point for correct bonuses
		if weapon:
			weaponName = weapon.GetClassname()
			if "silenced" in weaponName: # Silenced weapons
				point *= int(GEUtil.GetCVarValue("ass_silencedbonus"))
			elif "knife" in weaponName or "slappers" in weaponName: # Knife/Slapper weapons
				point *= int(GEUtil.GetCVarValue("ass_meleebonus"))
		if not killer:
			# World kill
			victim.IncrementScore(-1)
		elif victim.GetIndex() == killer.GetIndex():
			# Suicide
			killer.IncrementScore(-1)
		elif self.AssassinationTargets[killer.GetIndex()] == victim.GetIndex():
			# Assassinated your target.  Good job!
			killer.IncrementScore(point)
			GEUtil.HudMessage(killer, "You killed your target!", -1, -1, self.COLOUR_TARGET_MSG, 2.0)
			GEUtil.HudMessage(victim, "You have been assassinated!", -1, -1, self.COLOUR_TARGET_MSG, 2.0)
			self.IncrementBadKills(killer, -1)
		elif self.AssassinationTargets[victim.GetIndex()] == killer.GetIndex():
			# Killed your assassination.  Even better job!
			killer.IncrementScore(point*2)
			GEUtil.HudMessage(killer, "You killed your assassin!", -1, -1, self.COLOUR_ASSASSIN_MSG, 2.0)
			GEUtil.HudMessage(victim, "Your target got you instead!", -1, -1, self.COLOUR_ASSASSIN_MSG, 2.0)
			self.IncrementBadKills(killer, -1)
		else:
			# Bad player, don't do that!
			GEUtil.HudMessage(killer, "You killed the wrong person!", -1, -1, self.COLOUR_BADKILL_MSG, 2.0)
			killer.IncrementScore(-1)
			self.IncrementBadKills(killer, 1)
		# Finally, "kill" the target in the target list
		self.KillTarget(victim)
		
	def OnPlayerSpawn(self, player):
		self.DEBUG("Spawning %s" % (player.GetPlayerName()))
		if not player:
			return
			# Generate a new target if the player doesn't already have one (after round start)
		self.DEBUG("%s's target ID: %s" % (player.GetPlayerName(),self.AssassinationTargets[player.GetIndex()]))
		if self.AssassinationTargets[player.GetIndex()] == -1:
			self.GenerateNewTarget(player)
		
	def OnThink(self):
		if GEMPGameRules.GetNumActivePlayers() < 2:
			# Enter a state of waiting if there aren't enough players to continue
			GEUtil.ClientPrintAll(GEGlobal.HUD_PRINTCENTER, "Waiting For Players ...")
			self.WaitingForPlayers = True
			return
		elif self.WaitingForPlayers:
			# Enough players are in, give them targets!
			self.WaitingForPlayers = False
			GEMPRules.EndRound(False)
			return
		scoreLimit = int(GEUtil.GetCVarValue("ass_scorelimit"))
		for i in range(32):
			# Iterate through indexes and skip invalid ones
			if not GEUtil.IsValidPlayerIndex(i):
				continue
			player = GEUtil.GetMPPlayer(i)
			# Update radar entries here, when implemented here
			if GEUtil.GetTime() > self.NextDistUpdate:
			# Update player target texts
				self.TargetHUD(player, self.AssassinationTargets[i])
			if self.BadKillsTime[i] > 0:
				# Display a message to bad killers with punishment time remaining
				message = "Too many wrong kills! Punishment time: %i" % (self.BadKillsTime[i] - GEUtil.GetTime())
				GEUtil.ClientPrintPlayer(player, GEGlobal.HUD_PRINTCENTER, message)
			if GEUtil.GetTime() > self.BadKillsTime[i] and self.BadKillsTime[i] > 0:
				# Reset bad kill status after punishment time is up
				GEUtil.ClientPrintPlayer(player, GEGlobal.HUD_PRINTCENTER, "Punishment time over...")
				self.BadKills[player.GetIndex()] = 0
				self.BadKillsTime[player.GetIndex()] = 0
				player.SetDamageMultiplier(1.0)
			if scoreLimit != 0:
				# Check for possible win conditions
				if (player.GetMatchScore() + player.GetScore()) >=  scoreLimit:
					GEMPGameRules.EndMatch()
		if GEUtil.GetTime() > self.NextDistUpdate:
			# Generate next update time
			self.NextDistUpdate = GEUtil.GetTime() + self.DIST_UPDATE_INTERVAL
		
	def ShouldDoCustomDamage(self, victim, info, health, armor):
		if not victim:
			return health, armour
		attacker = GEPlayer.ToMPPlayer(info.GetAttacker())
		if not attacker:
			return health, armor
		self.DEBUG("Custom Damage: %s" % (attacker.GetName()))
		if self.BadKillsTime[attacker.GetIndex()] > 0:
			return 0, 0
		return health, armor
		
	def MassGenerateTargets(self):
		try:
			self.DEBUG("Mass generating targets")
			# Clear target list
			self.AvaliableTargets = []
			for i in range(32):
				# If the player is real, and is not spectating...
				if not GEUtil.IsValidPlayerIndex(i):
					continue
				if GEUtil.GetMPPlayer(i).GetTeamNumber() == GEGlobal.TEAM_SPECTATOR:
					continue
				# Add the player to the avaliable list
				self.AvaliableTargets.append(i)
			# Can't do shit all if there isn't enough players to generate targets!
			if len(self.AvaliableTargets) <= 1:
				self.DEBUG("Not enough players to generate targets")
				return
			self.DEBUG("All avaliable targets: %s" % (self.AvaliableTargets), 1)
			# Iterate over avaliable targets
			for i in self.AvaliableTargets:
				player = GEUtil.GetMPPlayer(i)
				# Make a copy of the avaliable targets and remove the current player from it
				indexes = copy(self.AvaliableTargets)
				indexes.remove(i)
				target = choice(indexes)
				# Set the player-only radar for player to display target, when implemented here
				self.AssassinationTargets[i] = target
				self.TargetHUD(player, target)
		except:
			self.DEBUG("An error has occured in the MassGenerateTargets function.", -1)
			self.DEBUG("%s" % (sys.exc_info()[0]), -1)
		
	def GenerateNewTarget(self, player):
		try:
			self.DEBUG("Generating new target for %s" % (player.GetPlayerName()))
			# If the player is real, or we're not waiting for more people
			if not player:
				return
			# Can't do anything if there's not enough players
			if len(self.AvaliableTargets) <= 1:
				self.DEBUG("Not enough players to generate target")
				return
			# Copy the avaliable targets and remove player from it
			indexes = copy(self.AvaliableTargets)
			indexes.remove(player.GetIndex())
			self.DEBUG("Target list for %s: %s" % (player.GetPlayerName(), indexes), 1)
			target = choice(indexes)
			# Set the player-only radar for player to display target, when implemented here
			self.AssassinationTargets[player.GetIndex()] = target
			self.TargetHUD(player, target)
		except:
			self.DEBUG("An error has occured in the GenerateNewTarget function.", -1)
			self.DEBUG("%s" % (sys.exc_info()[0]), -1)
		
	def AddTarget(self, player):
		try:
			self.DEBUG("Adding target: %s" % (player.GetPlayerName()))
			if not player:
				return
			i = player.GetIndex()
			# If the player is not already in the game, add him to it
			if not i in self.AvaliableTargets:
				self.AvaliableTargets.append(i)
				self.AvaliableTargets.sort()
			# Reset variables at the correct list indexies
			self.AssassinationTargets[i] = -1
			self.BadKills[i] = 0
			self.BadKillsTime[i] = 0
		except:
			self.DEBUG("An error has occured in the AddTarget function.", -1)
			self.DEBUG("%s" % (sys.exc_info()[0]), -1)
		
	def RemoveTarget(self, player):
		try:
			self.DEBUG("Removing target: %s" % (player.GetPlayerName()))
			# If the player is real, and is in the game already
			if not player:
				return
			i = player.GetIndex()
			if not i in self.AvaliableTargets:
				return
			# Remove the player, and reset variables
			self.AvaliableTargets.remove(i)
			self.AssassinationTargets[i] = -1
			self.BadKills[i] = 0
			self.BadKillsTime[i] = 0
			# Now "kill" the player so nobody still has him as their target
			self.KillTarget(player)
		except:
			self.DEBUG("An error has occured in the RemoveTarget function.", -1)
			self.DEBUG("%s" % (sys.exc_info()[0]), -1)
		
	def KillTarget(self, player):
		try:
			self.DEBUG("Killing target: %s" % (player.GetPlayerName()))
			if not player:
				return
			index = player.GetIndex()
			self.AssassinationTargets[index] = -1
			# Remove player-only radar for player, when implemented here
			GEUtil.RemoveHudProgressBarPlayer(player, 0)
			GEUtil.RemoveHudProgressBarPlayer(player, 1)
			# Go over each avaliable targets
			self.DEBUG("Iterating over avaliable targets.", 1)
			for i in self.AvaliableTargets:
				if not GEUtil.IsValidPlayerIndex(i):
					continue
				# Reselect targets for those who have the "killed" player
				if self.AssassinationTargets[i] == index:
					otherPlayer = GEUtil.GetMPPlayer(i)
					self.DEBUG("%s has %s as their target" % (otherPlayer.GetPlayerName(), player.GetPlayerName()))
					# Remove player-only radar entries here, when implemented here
					self.GenerateNewTarget(otherPlayer)
		except:
			self.DEBUG("An error has occured in the KillTarget function.", -1)
			self.DEBUG("%s" % (sys.exc_info()[0]), -1)
		
	def TargetHUD(self, player, target):
		# The usual player checking...
		if not player or not GEUtil.IsValidPlayerIndex(target):
			return
		# Get target player
		targetPlayer = GEUtil.GetMPPlayer(target)
		# Construct a dummy colour
		color = GEUtil.CColor(220,220,220,240)
		if targetPlayer.IsDead():
			# If the target is dead, make dummy values
			distance = 0
			distanceStr = "Dead"
			zStr = ""
		else: 
			# If he is alive, preform the correct calculations
			distance = int(GEUtil.DistanceBetween(player, targetPlayer) / 4.8768)
			distanceStr = "%im" % (distance)
			playerZ = GEUtil.Vector.__getitem__(player.GetAbsOrigin(), 2)
			targetZ = GEUtil.Vector.__getitem__(targetPlayer.GetAbsOrigin(), 2)
			# Adjust Z String for height variance
			if (playerZ + 96) <= targetZ:
				zStr = " Above"
			elif (playerZ - 96) >= targetZ:
				zStr = " Below"
			else:
				zStr = ""
				# Construct a colour based on the distance away
				blue = int(240 - 240 * (100 - min([100,max([distance-105,0])])) / 100)
				green = int(240 - 240 * (100 - min([100,max([distance-5,0])])) / 100)
				color = GEUtil.CColor(240,green,blue,240)
		# Grab the character name from the dictionary
		character = self.CHARACTER_DICT[targetPlayer.GetPlayerModel().lower()]
		# Now update the text if the distances have changed
		if self.LastDistance[player.GetIndex()] != distance:
			# Update player radars here
			self.LastDistance[player.GetIndex()] = distance
			GEUtil.RemoveHudProgressBarPlayer(player, 0)
			GEUtil.RemoveHudProgressBarPlayer(player, 1)
			# Top string is for player name and character
			message1 = "%s (%s)" % (targetPlayer.GetPlayerName(), character)
			# Bottom string is for distance
			message2 = "%s%s" % (distanceStr, zStr)
			GEUtil.InitHudProgressBarPlayer(player, 0, message1, 0, 1.0, -1, 0.73, 120, 10, GEUtil.CColor(240,240,240,240))
			GEUtil.InitHudProgressBarPlayer(player, 1, message2, 0, 1.0, -1, 0.78, 120, 10, color)
		
	def IncrementBadKills(self, player, val=0):
		try:
			self.DEBUG("IncrementBadKills: %s, %s" % (player.GetPlayerName(), val))
			if not player:
				return
			treshhold = int(GEUtil.GetCVarValue("ass_badkills"))
			time = int(GEUtil.GetCVarValue("ass_punishtime"))
			self.DEBUG("Bad kill settings: %i, %i" % (threshhold, time), 1)
			# If bad kill punishment is enabled through CVars
			if treshold > 0 and time > 0:
				i = player.GetIndex()
				# Add value to count
				self.BadKills[i] += val
				if self.BadKills[i] < 0:
					# Value correction
					self.BadKills[i] = 0
				elif self.BadKills[i] >= threshhold:
					# Punish the player if above the threshhold
					self.BadKillsTime[i] = GEUtil.GetTime() + time
					self.BadKills[i] = 0
		except:
			self.DEBUG("An error has occured in the IncrementBadKills function.", -1)
			self.DEBUG("%s" % (sys.exc_info()[0]), -1)
		
	def DEBUG(self, string, verbosity=0):
		if int(GEUtil.GetCVarValue("ass_debug")) > verbosity:
			GEUtil.ClientPrintAll(GEGlobal.HUD_PRINTTALK, string)